#include <stdio.h>
#include <stdbool.h>
#include "mock.h"
#include "mock_cfg.h"

int func1_hook1(std::unique_ptr<const int> pArg, va_list args)
{
    return 123;
}

int func1_hook2(std::unique_ptr<const int> pArg, va_list args)
{
    return MOCK_FUNCTIONS(hook).func1.shadowCall(std::move(pArg), args);
}

int g_func2Hook1Called = 0;
void func2_hook1(const coords_t *pCoords, const int numCoords)
{
    g_func2Hook1Called++;
}

void func2_hook2(const coords_t *pCoords, const int numCoords)
{
    MOCK_FUNCTIONS(hook).func2.shadowCall(pCoords, numCoords);
}

int g_func3Hook1Called = 0;
void func3_hook1(coords_t *pRetCoords, const int numCoords)
{
    g_func3Hook1Called++;
}

void func3_hook2(coords_t *pRetCoords, const int numCoords)
{
    MOCK_FUNCTIONS(hook).func3.shadowCall(pRetCoords, numCoords);
    /* overwrite the coordinates to make sure it has priority */
    pRetCoords[0].x = 0.1f;
    pRetCoords[0].y = 0.2f;
}

int g_intdivHook1Called = 0;
int intdiv_hook1(coords_t *pImplicitThis, int *pRemainder)
{
    g_intdivHook1Called++;
    
    float div = (pImplicitThis->x / pImplicitThis->y);
    int ret = (int)div;
    if (pRemainder) {
        *pRemainder = (int)(pImplicitThis->x - (pImplicitThis->y * (float)ret));
    }
    return ret;
}

int intdiv_hook2(coords_t *pImplicitThis, int *pRemainder)
{
    float ret = MOCK_FUNCTIONS(hook.coords).intdiv.shadowCall(pImplicitThis, pRemainder);
    /* overwrite the remainder */
    *pRemainder = 101.0f;
    return ret;
}

void test_func1(void)
{
    START_TEST;
    const int *val1 = new int(111);

    MOCK_FUNCTIONS(hook).func1.setHook(func1_hook1);
    int res = func1(std::unique_ptr<const int>(val1));
    assert_eq_fatal(res, 123);

    const int *val2 = new int(222);
    MOCK_FUNCTIONS(hook).func1.reset();
    res = func1(std::unique_ptr<const int>(val2));
    assert_eq_fatal(res, 0);

    const int *val3 = new int(333);
    MOCK_FUNCTIONS(hook).func1.setHook(func1_hook2);
    res = func1(std::unique_ptr<const int>(val3));
    assert_eq_fatal(res, 0);

    const int *val4 = new int(444);
    MOCK_FUNCTIONS(hook).func1.reset();
    MOCK_FUNCTIONS(hook).func1.setHook(func1_hook2);
    MOCK_FUNCTIONS(hook).func1.expect(std::unique_ptr<const int>(val4));
    MOCK_FUNCTIONS(hook).func1.andReturn(678);
    res = func1(std::unique_ptr<const int>(val4));
    assert_eq_fatal(res, 678);
    END_TEST;
    confirmSuccessOrDie();
}

void coordAssign(coords_t *pLHS, coords_t *pRHS, const unsigned long count)
{
    size_t i = 0;
    for (i = 0; i < count; i++) {
        pLHS[i].x = pRHS[i].x;
        pLHS[i].y = pRHS[i].y;
    }
}

bool coordCheck(coords_t *pLHS, coords_t *pRHS, const unsigned long count)
{
    bool res = true;
    size_t i = 0;
    for (i = 0; i < count; i++) {
        res = res && (pLHS[i].x == pRHS[i].x) && (pLHS[i].y == pRHS[i].y);
    }
    return res;
}

void test_func2(void)
{
    const coords_t testCoords[] = {
        {1, 2},
        {3, 4},
        {5, 6}
    };
    const int NUM_COORDS = sizeof(testCoords) / sizeof(testCoords[0]);

    START_TEST;
    MOCK_FUNCTIONS(hook).func2.setHook(func2_hook1);
    MOCK_FUNCTIONS(hook).func2.expect(testCoords, NUM_COORDS, NUM_COORDS);
    func2(testCoords, NUM_COORDS);
    assert_eq_fatal(g_func2Hook1Called, 1);

    MOCK_FUNCTIONS(hook).func2.reset();
    MOCK_FUNCTIONS(hook).func2.expect(testCoords, NUM_COORDS, NUM_COORDS);
    
    func2(testCoords, NUM_COORDS);
    assert_eq_fatal(g_func2Hook1Called, 1);

    /* moving on to hook 2, which will call the shadow function */
    MOCK_FUNCTIONS(hook).func2.reset();
    MOCK_FUNCTIONS(hook).func2.setHook(func2_hook2);
    func2(testCoords, NUM_COORDS);
    assert_eq_fatal(g_func2Hook1Called, 1);
    END_TEST;
    confirmSuccessOrDie();

    START_TEST;
    const coords_t badTestCoords[] = {
        {1, 2},
        {3, 4},
        {5, 7}
    };
    MOCK_FUNCTIONS(hook).func3.setCompare(coordCheck, NULL);
    MOCK_FUNCTIONS(hook).func2.expect(testCoords, NUM_COORDS, NUM_COORDS);
    func2(badTestCoords, NUM_COORDS);
    END_TEST;
    confirmFailureOrDie();

    START_TEST;
    MOCK_FUNCTIONS(hook).func2.reset();
    MOCK_FUNCTIONS(hook).func2.setHook(func2_hook2);
    
    MOCK_FUNCTIONS(hook).func2.expect(testCoords, NUM_COORDS, NUM_COORDS);
    func2(testCoords, NUM_COORDS);
    END_TEST;
    confirmSuccessOrDie();
}

void test_func3(void)
{
    coords_t testCoordsIn[] = {
        {1, 2},
        {3, 4},
        {5, 6}
    };
    coords_t testCoordsOut1[] = {
        {10, 20},
        {30, 40},
        {50, 60}
    };
    coords_t testCoordsOut2[] = {
        {100, 200},
        {300, 400},
        {500, 600}
    };
    const int NUM_COORDS = sizeof(testCoordsIn) / sizeof(testCoordsIn[0]);

    START_TEST;
    MOCK_FUNCTIONS(hook).func3.setHook(func3_hook1);
    MOCK_FUNCTIONS(hook).func3.expect(testCoordsIn, NUM_COORDS, NUM_COORDS);
    MOCK_FUNCTIONS(hook).func3.andReturn(testCoordsOut1, NUM_COORDS);
    func3(testCoordsIn, NUM_COORDS);
    assert_eq_fatal(g_func3Hook1Called, 1);
    
    assert_eq_fatal(testCoordsIn[2].x, 5.0f);

    MOCK_FUNCTIONS(hook).func3.reset();
    
    MOCK_FUNCTIONS(hook).func3.expect(testCoordsIn, NUM_COORDS, NUM_COORDS);
    MOCK_FUNCTIONS(hook).func3.setAssign(coordAssign);
    MOCK_FUNCTIONS(hook).func3.andReturn(testCoordsOut1, NUM_COORDS);
    func3(testCoordsIn, NUM_COORDS);
    assert_eq_fatal(g_func3Hook1Called, 1);
    
    assert_eq_fatal(testCoordsIn[2].x, 50.0f);

    /* moving on to hook 2, which will call the shadow function */
    MOCK_FUNCTIONS(hook).func3.reset();
    MOCK_FUNCTIONS(hook).func3.setHook(func3_hook2);
    MOCK_FUNCTIONS(hook).func3.expect(testCoordsIn, NUM_COORDS, NUM_COORDS);
    MOCK_FUNCTIONS(hook).func3.setAssign(coordAssign);
    MOCK_FUNCTIONS(hook).func3.andReturn(testCoordsOut2, NUM_COORDS);
    func3(testCoordsIn, NUM_COORDS);
    assert_eq_fatal(g_func3Hook1Called, 1);
    
    assert_eq_fatal(testCoordsIn[2].x, 500.0f);
    
    assert_eq_fatal(testCoordsIn[0].x, 0.1f);
    END_TEST;
    confirmSuccessOrDie();

    START_TEST;
    coords_t badTestCoords[] = {
        {100, 200},
        {300, 400},
        {500, 700}
    };
    MOCK_FUNCTIONS(hook).func3.reset();
    MOCK_FUNCTIONS(hook).func3.setHook(func3_hook2);
    MOCK_FUNCTIONS(hook).func3.setCompare(coordCheck, NULL);
    MOCK_FUNCTIONS(hook).func3.expect(testCoordsIn, NUM_COORDS, NUM_COORDS);
    func3(badTestCoords, NUM_COORDS);
    END_TEST;
    confirmFailureOrDie();
}

void test_intdiv(void)
{
    int res = 0;
    int remain = 0;
    int remainReturn = 0;
    coords_t uut1 = { 3.0, 2.0 };

    START_TEST;

    MOCK_FUNCTIONS(hook.coords).intdiv.setHook(intdiv_hook1);
    res = uut1.intdiv(&remain);
    assert_eq_fatal(res, 1);
    assert_eq_fatal(remain, 1);

    remain = 0;
    MOCK_FUNCTIONS(hook.coords).intdiv.reset();
    res = uut1.intdiv(&remain);
    assert_eq_fatal(res, 0);
    assert_eq_fatal(remain, remain);

    remainReturn = 444;
    MOCK_FUNCTIONS(hook.coords).intdiv.expect(&uut1);
    MOCK_FUNCTIONS(hook.coords).intdiv.andReturn(333, &remainReturn, 1);
    
    
    MOCK_FUNCTIONS(hook.coords).intdiv.setHook(intdiv_hook1);
    res = uut1.intdiv(&remain);
    assert_eq_fatal(res, 1);
    assert_eq_fatal(remain, 1);

    
    MOCK_FUNCTIONS(hook.coords).intdiv.reset();
    MOCK_FUNCTIONS(hook.coords).intdiv.expect(&uut1);
    MOCK_FUNCTIONS(hook.coords).intdiv.andReturn(333, &remainReturn, 1);
    res = uut1.intdiv(&remain);
    assert_eq_fatal(res, 333);
    assert_eq_fatal(remain, 444);

    
    
    MOCK_FUNCTIONS(hook.coords).intdiv.reset();
    MOCK_FUNCTIONS(hook.coords).intdiv.expect(&uut1);
    MOCK_FUNCTIONS(hook.coords).intdiv.andReturn(333, &remainReturn, 1);
    MOCK_FUNCTIONS(hook.coords).intdiv.setHook(intdiv_hook2);
    res = uut1.intdiv(&remain);
    END_TEST;
    confirmSuccessOrDie();
    assert_eq_fatal(res, 333);
    
    assert_eq_fatal(remain, 101);

    
    START_TEST;
    MOCK_FUNCTIONS(hook.coords).intdiv.expect(NULL);
    MOCK_FUNCTIONS(hook.coords).intdiv.andReturn(555, &remainReturn, 1);
    res = uut1.intdiv(&remain);
    END_TEST;
    confirmFailureOrDie();
}

int main(int argc, char **argv)
{
    test_func1();
    test_func2();
    test_func3();
    test_intdiv();
    return 0;
}
