#!/bin/bash

################################################################################
# alltests.sh
#
# Run the battery of Mimicc integration tests.
#
# This script iterates over a given directory looking for integration tests and
# runs them one by one. More specifically it runs the sibling script
# 'runtest.sh' within the folder of each test case.
#
# There is one input: the path to the folder containing all test cases. The
# following options are also supported:
#    -p <N times>: run a profiler instead of the standard pass/fail test.
#                  The profiler will run each sub-test N times and record
#                  the runtimes, calculating an average at the end.
#    -s: Save temporaries. This will tell the script to save each mock.cpp and
#                          mock.h output as the tests are run.
#
# The best way to run this is to with a modified PATH environment that allows
# the runtest.sh script to pick up the mimicc + clang distribution. Example:
#
# PATH=~/mimicc-dist/bin:$PATH ./bin/alltests.sh
#
# Note that since runtest.sh is the primary executive for each test, you can
# also specify an alternate compiler for building and running the test
# cases alongside the mimicc mock generation. Example:
#
# PATH=~/mimicc-dist/bin:$PATH CC=gcc CXX=g++ ./bin/alltests.sh
#
# This command would cause each test case to build a mock.o object file for
# each test case header using mimicc, and then use gcc/g++ to build main.cpp
# and link each case before running.
#
# If CC and CC are not specified, the default is to use the clang/clang++
# binaries that live alongside mimicc in the distribution bin folder.
################################################################################

function error_exit() {
    echo "execution failed" 1>&2
    exit 1
}

GLOBAL_DELIMITER=""

avg_time() {
    #
    # usage: avg_time n command ...
    #
    n=$1; shift
    (($# > 0)) || return                   # bail if no command given
    THIS_TEST="$(basename "$2")"
    printf "$GLOBAL_DELIMITER{\"name\": \"$THIS_TEST\", "
    for ((i = 0; i < n; i++)); do
        { time -p "$@" &>/dev/null; } 2>&1 # ignore the output of the command
                                           # but collect time's output in stdout
    done | awk '
        /real/ { real = real + $2; nr++ }
        /user/ { user = user + $2; nu++ }
        /sys/  { sys  = sys  + $2; ns++}
        END    {
                 if (nr>0 && nu>0 && ns>0)
                    printf("\"results\": {\"real\": %f, \"user\": %f, \"sys\": %f}", real/nr, user/nu, sys/ns);
               }'
    printf "}"
}

#check if profiling is neabled
while getopts "p:s" OPTION; do
    case $OPTION in
    p)
        PROFILE_COUNT=$OPTARG
        ;;
    s)
        SAVE_TEMPS="True"
        ;;
    esac
done

SCRIPT_DIR="$( cd "$( dirname "${BASH_SOURCE[0]}" )" >/dev/null 2>&1 && pwd )"

# if no path to tests is provided, assume the tests are at ../tests
if [ "$#" -lt "1" ]; then
    if [ ! -d "$SCRIPT_DIR/../tests" ]; then
        echo "no test directory found"
        error_exit
    fi
    TEST_DIR="$( cd $SCRIPT_DIR/../tests && pwd )"
else
    TEST_DIR=${@:OPTIND:1}
fi

# And run the tests
TESTS=$(cd $TEST_DIR && find . ! -path . -type d | sort)
if [ ! -z $PROFILE_COUNT ]; then
    printf "{\n    [\n        "
fi
for TEST in $TESTS; do
    if [ ! -z $PROFILE_COUNT ]; then
        (cd $TEST_DIR/$TEST && avg_time $PROFILE_COUNT bash "$SCRIPT_DIR/runtest.sh" $(basename "$TEST") $SAVE_TEMPS)
        GLOBAL_DELIMITER=',\n        '
    else
        echo "Test '$TEST'"
        (cd $TEST_DIR/$TEST && bash "$SCRIPT_DIR/runtest.sh" $(basename "$TEST") $SAVE_TEMPS)
    fi
    (( "$?" != "0" )) && error_exit
done
if [ ! -z $PROFILE_COUNT ]; then
    printf "\n    ]\n}\n"
fi

exit 0
