#!/bin/bash
################################################################################
# runtest.sh
#
# Run a single mimicc integration test.
#
# Integration tests are comprised of a header file with either .h or .hpp
# extenstion, and a main file (main.c or main.cpp). The extension determines
# whether mimicc or mimicc++ is used.
#
# The script accepts one ordinary argument, which is the name of the test,
# which should match the header file being mocked. The name of the test can
# also just be the name of the header.
#
# The script also picks up the environment variables CC and CXX which can
# optionally override the default 'clang' and 'clang++' used for actually
# compiling and linking the final test. This allows you to test for example
# that object files generated by mimicc can correctly link and run with
# a main.cpp/test harness compiled with gcc or visual studio.
#
# Example:
#    ./runtest.sh inputs.hpp
#
#    ./runtest.sh inputs.hpp CC=gcc CXX=g++
#
# A test run produces three logs:
#    1. mimicc.log: STDIO and STDERR output from mimicc being run on the
#                    input header file
#    2. compiler.log: STDIO and STDERR output from the compile step that takes
#                     mimicc's output and links it with the compiled main.cpp
#    3. test.log: STDIO and STDERR output from running the test
#
# It is assumed this script is run within the directory of the test, the
# script makes no attempt to search for the input header file beyond CWD.
################################################################################
MIMICC_EXE=mimicc
CPP_STD="17"
C_STD="89"

# allow compiler to be overridden on the command line
if [ -z $CC ]; then
    CC=clang
fi
if [ -z $CXX ]; then
    CXX=clang++
fi

if [ "$#" -eq "0" ]; then
    echo "usage: $0 <name of test>" 1>&2
    exit 1
fi

INPUT_NAME=$1

# optional parameter:
SAVE_TEMPS=$2

function error_exit() {
    echo "Test '$1' FAILED"
    shift
    if [ -z $SAVE_TEMPS ]; then
        for FILE in "$@"; do
            rm -rf $FILE
        done
    fi
    exit 1
}

# make sure there weren't some random other files laying
# around with the same root name
if [ $(find . -name "$INPUT_NAME*" -type f | wc -l) -ne 1 ]; then
    echo "Found either zero ro more than one file with root name '$INPUT_NAME'."
    echo "Please clean up this directory and try again."
    error_exit $INPUT_NAME
fi
INPUT_FILE=$(find . -name "$INPUT_NAME*" -type f)
(( "$?" != "0" )) && error_exit $INPUT_NAME

FILE_EXT="${INPUT_FILE##*.}"

if [ -f IGNORE ]; then
    echo "IGNORE file found, skipping test '$INPUT_NAME'"
    exit 0
fi

COMPILER_EXTRA_ARGS="-std=c$C_STD"
CODE_EXT=".c"
COMPILER_EXE="$CC"
if [ $FILE_EXT == "hpp" ] || [ $FILE_EXT == "cpp" ]; then
    COMPILER_EXTRA_ARGS="-std=c++$CPP_STD"
    COMPILER_EXE="$CXX"
    CODE_EXT=".cpp"
    MIMICC_EXE="$MIMICC_EXE++"
fi

OBJ_DIR="$(mktemp -d)"
if [ ! -z $SAVE_TEMPS ]; then
    OBJ_DIR="./$INPUT_NAME-temps"
    mkdir -p $OBJ_DIR
fi
SOUT_ARG="--sout=$OBJ_DIR/mock$CODE_EXT"
HOUT_ARG="--hout=$OBJ_DIR/mock.h"
TMP_FILE_MOCK_OBJ="$OBJ_DIR/out.o"
TMP_FILE_MOCK_SRC="$OBJ_DIR/out$CODE_EXT"
TMP_FILE_EXE="$OBJ_DIR/test-exe"
COMPILER_EXTRA_ARGS="-I $OBJ_DIR -I ../ $COMPILER_EXTRA_ARGS"

$MIMICC_EXE $INPUT_FILE $HOUT_ARG $COMPILER_EXTRA_ARGS -c -o $TMP_FILE_MOCK_OBJ $SOUT_ARG > mimicc.log 2>&1
(( "$?" != "0" )) && error_exit $INPUT_NAME $OBJ_DIR

$COMPILER_EXE $COMPILER_EXTRA_ARGS $TMP_FILE_MOCK_OBJ main$CODE_EXT -o $TMP_FILE_EXE > compiler.log 2>&1
(( "$?" != "0" )) && error_exit $INPUT_NAME $OBJ_DIR

"$TMP_FILE_EXE" > test.log 2>&1
(( "$?" != "0" )) && error_exit $INPUT_NAME $OBJ_DIR

echo "SUCCESS"
exit 0
