

















#ifndef LLVM_ADT_OPTIONAL_H
#define LLVM_ADT_OPTIONAL_H

#include <memory>
#include <new>
#include <utility>

enum class NoneType { None = 1 };
const NoneType None = NoneType::None;

namespace optional_detail {

struct in_place_t {};


template <typename T, bool = std::is_trivially_copyable<T>::value>
class OptionalStorage {
  union {
    char empty;
    T value;
  };
  bool hasVal;

public:
  ~OptionalStorage() { reset(); }

  OptionalStorage() noexcept : empty(), hasVal(false) {}

  OptionalStorage(OptionalStorage const &other) : OptionalStorage() {
    if (other.hasValue()) {
      emplace(other.value);
    }
  }
  OptionalStorage(OptionalStorage &&other) : OptionalStorage() {
    if (other.hasValue()) {
      emplace(std::move(other.value));
    }
  }

  template <class... Args>
  explicit OptionalStorage(in_place_t, Args &&... args)
      : value(std::forward<Args>(args)...), hasVal(true) {}

  void reset() noexcept {
    if (hasVal) {
      value.~T();
      hasVal = false;
    }
  }

  bool hasValue() const noexcept { return hasVal; }

  T &getValue()  noexcept {
    return value;
  }
  T const &getValue() const  noexcept {
    return value;
  }

  template <class... Args> void emplace(Args &&... args) {
    reset();
    ::new ((void *)std::addressof(value)) T(std::forward<Args>(args)...);
    hasVal = true;
  }

  OptionalStorage &operator=(T const &y) {
    if (hasValue()) {
      value = y;
    } else {
      ::new ((void *)std::addressof(value)) T(y);
      hasVal = true;
    }
    return *this;
  }
  OptionalStorage &operator=(T &&y) {
    if (hasValue()) {
      value = std::move(y);
    } else {
      ::new ((void *)std::addressof(value)) T(std::move(y));
      hasVal = true;
    }
    return *this;
  }

  OptionalStorage &operator=(OptionalStorage const &other) {
    if (other.hasValue()) {
      if (hasValue()) {
        value = other.value;
      } else {
        ::new ((void *)std::addressof(value)) T(other.value);
        hasVal = true;
      }
    } else {
      reset();
    }
    return *this;
  }

  OptionalStorage &operator=(OptionalStorage &&other) {
    if (other.hasValue()) {
      if (hasValue()) {
        value = std::move(other.value);
      } else {
        ::new ((void *)std::addressof(value)) T(std::move(other.value));
        hasVal = true;
      }
    } else {
      reset();
    }
    return *this;
  }
};

template <typename T> class OptionalStorage<T, true> {
  union {
    char empty;
    T value;
  };
  bool hasVal = false;

public:
  ~OptionalStorage() = default;

  OptionalStorage() noexcept : empty{} {}

  OptionalStorage(OptionalStorage const &other) = default;
  OptionalStorage(OptionalStorage &&other) = default;

  OptionalStorage &operator=(OptionalStorage const &other) = default;
  OptionalStorage &operator=(OptionalStorage &&other) = default;

  template <class... Args>
  explicit OptionalStorage(in_place_t, Args &&... args)
      : value(std::forward<Args>(args)...), hasVal(true) {}

  void reset() noexcept {
    if (hasVal) {
      value.~T();
      hasVal = false;
    }
  }

  bool hasValue() const noexcept { return hasVal; }

  T &getValue()  noexcept {
    return value;
  }
  T const &getValue() const  noexcept {
    return value;
  }

  template <class... Args> void emplace(Args &&... args) {
    reset();
    ::new ((void *)std::addressof(value)) T(std::forward<Args>(args)...);
    hasVal = true;
  }

  OptionalStorage &operator=(T const &y) {
    if (hasValue()) {
      value = y;
    } else {
      ::new ((void *)std::addressof(value)) T(y);
      hasVal = true;
    }
    return *this;
  }
  OptionalStorage &operator=(T &&y) {
    if (hasValue()) {
      value = std::move(y);
    } else {
      ::new ((void *)std::addressof(value)) T(std::move(y));
      hasVal = true;
    }
    return *this;
  }
};

} 

template <typename T> class Optional {
  optional_detail::OptionalStorage<T> Storage;

public:
  using value_type = T;

  constexpr Optional() {}
  constexpr Optional(NoneType) {}

  Optional(const T &y) : Storage(optional_detail::in_place_t{}, y) {}
  Optional(const Optional &O) = default;

  Optional(T &&y) : Storage(optional_detail::in_place_t{}, std::move(y)) {}
  Optional(Optional &&O) = default;

  Optional &operator=(T &&y) {
    Storage = std::move(y);
    return *this;
  }
  Optional &operator=(Optional &&O) = default;

  
  template <typename... ArgTypes> void emplace(ArgTypes &&... Args) {
    Storage.emplace(std::forward<ArgTypes>(Args)...);
  }

  static inline Optional create(const T *y) {
    return y ? Optional(*y) : Optional();
  }

  Optional &operator=(const T &y) {
    Storage = y;
    return *this;
  }
  Optional &operator=(const Optional &O) = default;

  void reset() { Storage.reset(); }

  const T *getPointer() const { return &Storage.getValue(); }
  T *getPointer() { return &Storage.getValue(); }
  const T &getValue() const  { return Storage.getValue(); }
  T &getValue()  { return Storage.getValue(); }

  explicit operator bool() const { return hasValue(); }
  bool hasValue() const { return Storage.hasValue(); }
  const T *operator->() const { return getPointer(); }
  T *operator->() { return getPointer(); }
  const T &operator*() const  { return getValue(); }
  T &operator*()  { return getValue(); }

  template <typename U>
  constexpr T getValueOr(U &&value) const  {
    return hasValue() ? getValue() : std::forward<U>(value);
  }

  
  template <class Function>
  auto map(const Function &F) const
      -> Optional<decltype(F(getValue()))> {
    if (*this) return F(getValue());
    return None;
  }
};

#endif 
